window.app = {
    data: [],
    serverTimeOffset: 0,
    updateInterval: null,
    tempData: {},

    init() {
        this.load();
        this.bindEvents();
        this.startHeartbeatChecks();
    },

    bindEvents() {
        const searchInput = document.getElementById('searchInput');
        if (searchInput) {
            searchInput.addEventListener('input', e => this.search(e.target.value));
        }
    },

    post(payload) {
        return fetch('pages/api.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify(payload)
        })
        .then(r => r.json())
        .catch(() => ({success:false}));
    },

    load() {
    this.post({ action: 'load_all' }).then(res => {
        if (!res?.success) return;
        this.processData(res);
        this.render();
        this.refreshSelects();
    });
},

refreshOnly() {
    this.post({ action: 'load_all' }).then(res => {
        if (!res?.success) return;
        this.processData(res);
        this.updateVMStatuses();
    });
},


startHeartbeatChecks() {
    if (this.updateInterval) {
        clearInterval(this.updateInterval);
    }

    this.updateInterval = setInterval(() => {
        this.refreshOnly();
    }, 30000);
},

fetchBalance(vmId) {
    const btn = document.getElementById(`fetch-btn-${vmId}`);
    if (btn) {
        btn.disabled = true;
        btn.innerText = 'Fetching...';
    }
    this.post({
        action: 'send_command',
        vm_id: vmId,
        command: 'FETCH_BALANCE'
    }).then(res => {
        if (res?.success) {
            setTimeout(() => {
                this.refreshOnly();
                if (btn) {
                    btn.disabled = false;
                    btn.innerText = 'Fetch Balance';
                }
            }, 15000);
        } else {
            if (btn) {
                btn.disabled = false;
                btn.innerText = 'Fetch Balance';
            }
            alert('Fetch failed');
        }
    });
},




getVmStatusData(v) {
    let isOnline = false;
    if (v.last_seen) {
        const hbTime = new Date(v.last_seen.replace(/-/g, '/') + ' UTC').getTime();
        const diff = (Date.now() - this.serverTimeOffset - hbTime) / 1000;
        if (diff <= 1800) isOnline = true; // 30 minutes
    }

    let runText = 'OFFLINE';
    let runClass = 'text-danger';

    if (v.db_status && v.db_status.toLowerCase() === 'blocked') {
        runText = 'BLACKLIST';
        runClass = 'text-danger';
    } else if (isOnline) {
        runText = 'RUNNING';
        runClass = 'text-success';
    }

    return {
        badgeText: isOnline ? 'ONLINE' : 'OFFLINE',
        badgeClass: isOnline ? 'bg-success' : 'bg-danger',
        runText,
        runClass,
        balance: v.balance || '$0.00',
        traffic: v.traffic || '0B'
    };
},

updateVMStatuses() {
    this.data.forEach(g => {
        g.rdps?.forEach(r => {
            r.vms?.forEach(v => {
                const s = this.getVmStatusData(v);
                const badge = document.getElementById(`v-badge-${v.id}`);
                const runEl = document.getElementById(`v-run-${v.id}`);
                const balEl = document.getElementById(`v-bal-${v.id}`);
                const trafEl = document.getElementById(`v-traf-${v.id}`);
                if (badge) {
                    badge.className = `badge ${s.badgeClass}`;
                    badge.innerText = s.badgeText;
                }
                if (runEl) {
                    runEl.innerText = s.runText;
                    runEl.className = s.runClass;
                }
                if (balEl) balEl.innerText = s.balance;
                if (trafEl) trafEl.innerText = s.traffic;
            });
        });
    });
},

 

    processData(res) {
        if (!res?.data?.data || !Array.isArray(res.data.data)) return;
    
        this.data = res.data.data;
    
        this.data.forEach(g => {
            if (!g.rdps) g.rdps = [];
            g.rdps.forEach(r => {
                if (!r.vms) r.vms = [];
            });
        });
    
        if (res.data.server_time) {
            const serverMs = new Date(res.data.server_time.replace(/-/g, '/') + ' UTC').getTime();
            this.serverTimeOffset = Date.now() - serverMs;
        }
    },

    search(term) {
        document.querySelectorAll('.searchable').forEach(el => {
            el.style.display = term ? 
                (el.textContent.toLowerCase().includes(term.toLowerCase()) ? '' : 'none') : 
                '';
        });
    },


 


    bulkDelete(type) {
        const selectors = {
            vm: { selector: '.vm-check:checked', action: 'delete_vms' },
            rdp: { selector: '.rdp-check:checked', action: 'delete_rdps' },
            group: { selector: '.group-check:checked', action: 'delete_groups' }
        };

        const config = selectors[type];
        if (!config) return;

        const ids = [...document.querySelectorAll(config.selector)].map(cb => cb.value);

        if (!ids.length) {
            alert('Select at least one ' + type);
            return;
        }

        if (!confirm(`Delete ${ids.length} ${type}(s)?`)) return;

        this.post({ action: config.action, ids }).then(res => {
            if (res?.success) {
                ids.forEach(id => {
                    document.querySelectorAll(`[value="${id}"]`).forEach(el => el.closest('.card')?.remove());
                });
            } else {
                alert('Delete failed');
            }
        });
    },

  
    restartVM(vmId) {
        this.post({ action: 'send_command', vm_id: vmId, command: 'RESTART_VM' })
            .then(res => { if (res?.success) alert('VM restart command sent'); });
    },

 
render() {
    const nav = document.getElementById('navTree');
    const main = document.getElementById('mainContent');
    if (!nav || !main) return;
    
    nav.innerHTML = '';
    main.innerHTML = '';
    this.tempData = {};

    this.data.forEach(g => {
        if (!g.id || g.id <= 0) return;
        const gNavId = `nav-g-${g.id}`;
        nav.innerHTML += `
            <li class="list-group-item bg-dark text-white searchable" style="cursor:pointer"
                onclick="document.getElementById('${gNavId}').classList.toggle('d-none')">
                ${g.name}
            </li>
            <ul id="${gNavId}" class="list-group list-group-flush d-none ms-3"></ul>
        `;
        
        const gNav = document.getElementById(gNavId);
        let groupHTML = `
            <div class="card mb-4 bg-dark text-white searchable">
              <div class="card-header d-flex justify-content-between align-items-center">
                <div>
                  <input type="checkbox" class="group-check me-2" value="${g.id}">
                  <strong>${g.name}</strong>
                </div>
              </div>
              <div class="card-body">`;

        g.rdps?.forEach(r => {
            const rNavId = `nav-r-${r.id}`;
            gNav.innerHTML += `
                <li class="list-group-item bg-black text-info searchable" style="cursor:pointer" 
                    onclick="event.stopPropagation(); document.getElementById('${rNavId}').classList.toggle('d-none')">
                    ${r.name}
                </li>
                <ul id="${rNavId}" class="list-group list-group-flush d-none ms-3"></ul>`;
            
            const rNav = document.getElementById(rNavId);
            groupHTML += `
                <div class="card bg-black text-white mb-3 searchable">
                  <div class="card-header d-flex justify-content-between align-items-center">
                    <div>
                      <input type="checkbox" class="rdp-check me-2" value="${r.id}">
                      <span>${r.name}</span>
                    </div>
                    <div class="d-flex gap-2">
                      <button class="btn btn-sm btn-outline-success"
                        onclick="app.openModal('vm', ${g.id}, ${r.id})">+ VM</button>
                    </div>
                  </div>
                  <div class="card-body">`;

            r.vms?.forEach(v => {
                const storeKey = `vm_${v.id}`;
                this.tempData[storeKey] = { v, r, g };
                rNav.innerHTML += `<li class="list-group-item bg-black text-success small searchable">${v.name}</li>`;

                let isOnline = false;
                if (v.last_seen) {
                    const hbTime = new Date(v.last_seen.replace(/-/g, '/') + ' UTC').getTime();
                    const diff = (Date.now() - this.serverTimeOffset - hbTime) / 1000;
                    if (diff <= 1800) isOnline = true;
                }

                let runText = 'OFFLINE';
                let runClass = 'text-danger';

                if (v.db_status && v.db_status.toLowerCase() === 'blocked') {
                    runText = 'BLACKLIST';
                    runClass = 'text-danger';
                } else if (isOnline) {
                    runText = 'RUNNING';
                    runClass = 'text-success';
                }

                const badgeText = isOnline ? 'ONLINE' : 'OFFLINE';
                const badgeClass = isOnline ? 'bg-success' : 'bg-danger';
                const auto = v.earnapp_autostart == 1;
                const paused = v.earnapp_paused == 1;

                groupHTML += `
                <div class="card mb-4 bg-black text-white border border-secondary rounded-4 searchable">
                  <div class="card-body d-flex justify-content-between align-items-center">
                    <div class="left-card d-flex gap-4">
                      <div class="status1">
                        <span id="v-badge-${v.id}" class="badge ${badgeClass}">${badgeText}</span>
                        <div class="text-danger fw-bold mt-2 d-flex align-items-center gap-2">
                          <input type="checkbox" class="vm-check" value="${v.id}">
                          <span>${v.name} - ${v.id}</span>
                        </div>
                        <div class="small">${v.current_ip || ''}</div>
                      </div>
                      <div class="border border-secondary rounded p-3 middle">
                        <div class="small">
                          <span id="link-${v.id}" class="d-none">https://earnapp.com/dashboard/link/sdk-win-${v.id}</span>
                          <button class="btn btn-sm btn-link text-danger p-0" onclick="app.copyToClipboard('link-${v.id}', true)">Copy Link</button>
                        </div>
                        <div class="small mt-2">
                          Run: <span id="v-run-${v.id}" class="${runClass}">${runText}</span>
                          | Auto: <span id="v-auto-${v.id}" class="${auto ? 'text-success' : 'text-danger'}">${auto ? 'ON' : 'OFF'}</span>
                          | Pause: <span id="v-pause-${v.id}" class="${paused ? 'text-danger' : 'text-success'}">${paused ? 'YES' : 'NO'}</span>
                        </div>
                        <span id="v-bal-${v.id}" class="badge bg-warning text-dark mt-2">${v.balance || '$0.00'}</span>
                        <div class="small text-info mt-1">
                          Traffic: <span id="v-traf-${v.id}">${v.traffic || '0B'}</span>
                        </div>
                      </div>
                    </div>
                    <div class="d-flex flex-column gap-2" style="width:140px">
                      <button class="btn btn-outline-light btn-sm" onclick="app.copyToClipboardText('sdk-win-${v.id}')">Copy Token</button>
                      <button id="fetch-btn-${v.id}" class="btn btn-outline-success btn-sm" onclick="app.fetchBalance(${v.id})">Fetch Balance</button>
                      <button class="btn btn-outline-warning btn-sm" onclick="app.restartVM(${v.id})">Restart App</button>
                      <button class="btn btn-outline-info btn-sm" onclick="app.openVmAgentCommands('${storeKey}')">Terminal</button>
                    </div>
                  </div>
                </div>`;
            });
            groupHTML += `</div></div>`;
        });
        groupHTML += `</div></div>`;
        main.innerHTML += groupHTML;
    });
},

    refreshSelects() {
        const selects = ['rdpGroupSelect', 'vmGroupSelect'];
        selects.forEach(id => {
            const sel = document.getElementById(id);
            if (!sel) return;
            sel.innerHTML = '<option value="">Select Group</option>';
            this.data.forEach(g => sel.innerHTML += `<option value="${g.id}">${g.name}</option>`);
        });
    },

    loadRDPsForVM(selectedRdp = null) {
        const gid = document.getElementById('vmGroupSelect')?.value;
        const sel = document.getElementById('vmRdpSelect');
        if (!gid || !sel) return;
        
        sel.innerHTML = '<option value="">Select RDP</option>';
        const g = this.data.find(x => x.id == gid);
        
        g?.rdps?.forEach(r => {
            const o = document.createElement('option');
            o.value = r.id;
            o.textContent = r.name;
            if (selectedRdp == r.id) o.selected = true;
            sel.appendChild(o);
        });
    },

    openModal(type, groupId = null, rdpId = null) {
        const el = document.getElementById('modal-' + type);
        if (!el) return;
        
        document.getElementById('overlay').style.display = 'block';
        el.style.display = 'flex';
        
        if (type === 'vm' && groupId) {
            const gSel = document.getElementById('vmGroupSelect');
            if (gSel) { gSel.value = groupId; this.loadRDPsForVM(rdpId); }
        }
    },

    closeModal() {
        document.getElementById('overlay').style.display = 'none';
        document.querySelectorAll('.modal').forEach(m => m.style.display = 'none');
    },

    postAction(action, data) {
        return this.post({ action, ...data }).then(res => {
            if (res?.success) {
                this.closeModal();
                this.load();  
            } else {
                alert('Operation failed');
            }
        });
    },

    addGroup() {
        const el = document.getElementById('groupInput');
        if (el) this.postAction('add_group', { name: el.value });
    },

    addRDP() {
        const nameEl = document.getElementById('rdpInput');
        const groupEl = document.getElementById('rdpGroupSelect');
        
        if (!nameEl || !groupEl) {
            alert('RDP modal not loaded properly');
            return;
        }

        if (!groupEl.value) {
            alert('Please select a group');
            return;
        }

        this.postAction('add_rdp', {
            name: nameEl.value,
            group_id: groupEl.value,
            ip: document.getElementById('rdpIp')?.value || '',
            username: document.getElementById('rdpUsername')?.value || ''
        });
    },

    addVM() {
        this.postAction('add_vm', {
            name: document.getElementById('vmInput').value,
            rdp_id: document.getElementById('vmRdpSelect').value
        });
    },

    openVmAgentCommands(storeKey) {
        const data = this.tempData[storeKey];
        if (!data) return;
        
        const { v, r, g } = data;
        this.currentTerminalData = { vmId: v.id, vmName: v.name, groupName: g.name, rdpName: r.name };
        
        ['terminalTitle', 'displayVmId', 'displayVmName', 'displayGroup', 'displayRdp'].forEach((id, i) => {
            const el = document.getElementById(id);
            const texts = [`Terminal - ${v.name}`, v.id, v.name, g.name, r.name];
            if (el) el.innerText = texts[i];
        });
        
        this.setTerminalMode('install');
        this.updateTerminalCommands();
        
        const modal = document.getElementById('terminalModal');
        if (modal) {
            document.getElementById('overlay').style.display = 'block';
            modal.style.display = 'flex';
        }
    },

    setTerminalMode(mode) {
        document.querySelectorAll('#terminalModal [data-mode]').forEach(b => b.classList.remove('active'));
        document.querySelector(`#terminalModal [data-mode="${mode}"]`)?.classList.add('active');
        
        ['install', 'uninstall', 'custom'].forEach(s => {
            const el = document.getElementById(`${s}Section`);
            if (el) el.style.display = (mode === s ? 'block' : 'none');
        });
    },

    updateTerminalCommands() {
        if (!this.currentTerminalData) return;
    
        const base = window.location.origin + window.location.pathname.replace('/index.php','').replace(/\/$/,'');
        const { vmId, vmName, groupName, rdpName } = this.currentTerminalData;
    
        const setVal = (id, val) => {
            const el = document.getElementById(id);
            if (el) el.value = val;
        };
    
        setVal(
    'installCmd',
    `irm "${base}/static/install.ps1" -OutFile i.ps1; .\\i.ps1 -VMId ${vmId} -UnitId "${vmName}" -Group "${groupName}" -RdpNickname "${rdpName}" -BackendUrl "${base}"`
);

    
        setVal(
            'uninstallCmd',
            `irm "${base}/static/uninstall.ps1" | iex`
        );
    },
    

    copyToClipboard(id, isText = false) {
        const el = document.getElementById(id);
        if (el) navigator.clipboard.writeText(isText ? el.innerText : el.value);
    },

    copyToClipboardText(text) { 
        navigator.clipboard.writeText(text); 
    },

    sendToVM() {
        const cmd = document.getElementById('customCmd').value;
        if (cmd && this.currentTerminalData) {
            this.post({ 
                action: 'send_command', 
                vm_id: this.currentTerminalData.vmId, 
                command: cmd 
            }).then(res => {
                if (res?.success) alert('Command Sent Successfully');
            });
            document.getElementById('customCmd').value = '';
        }
    }
};

document.addEventListener('DOMContentLoaded', () => window.app.init());