<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['user_id'])) {
    header("Location: ./login.php");
    exit;
}

require_once __DIR__ . '/../db.php';

$stmt = $conn->prepare("SELECT username, avatar FROM users WHERE id = ? LIMIT 1");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$stmt->bind_result($username, $avatar);
$stmt->fetch();
$stmt->close();

$user = ['username' => $username, 'avatar' => $avatar];
$currentUsername = $user['username'] ?? 'Unknown';
$avatar = $user['avatar'] ?? '';
$avatarUrl = $avatar ? "/earnapp/uploads/avatars/".$avatar : "/earnapp/assets/img/default-avatar.png";
?>

<style>
  @import url('https://fonts.googleapis.com/css2?family=Syne:wght@400;600;700;800&family=DM+Sans:wght@300;400;500&display=swap');

  :root {
    --crimson: #c0392b;
    --crimson-bright: #e74c3c;
    --crimson-glow: rgba(192, 57, 43, 0.25);
    --surface-0: #0a0a0a;
    --surface-1: #111111;
    --surface-2: #181818;
    --surface-3: #222222;
    --border: rgba(192, 57, 43, 0.2);
    --border-hover: rgba(192, 57, 43, 0.55);
    --text-primary: #f0f0f0;
    --text-secondary: #888;
    --text-muted: #555;
  }

  .settings-wrapper {
    font-family: 'DM Sans', sans-serif;
    background: var(--surface-0);
    min-height: 100vh;
    padding: 2rem 1.5rem;
  }

  /* ── Stats wrapper ── */
  #top-stats-wrapper {
    margin-bottom: 2.5rem;
  }

  /* ── Page header ── */
  .settings-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 2.5rem;
  }

  .settings-header-icon {
    width: 46px;
    height: 46px;
    background: linear-gradient(135deg, var(--crimson), #7b1a13);
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.1rem;
    color: #fff;
    box-shadow: 0 0 20px var(--crimson-glow);
    flex-shrink: 0;
  }

   

  .settings-header p {
    margin: 0;
    font-size: 0.8rem;
    color: var(--text-secondary);
  }

  /* ── Flash ── */
  .settings-flash {
    border-radius: 10px;
    border: 1px solid;
    padding: 0.85rem 1.25rem;
    margin-bottom: 1.5rem;
    font-size: 0.875rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
  }
  .settings-flash.success { border-color: #1a6644; background: rgba(26,102,68,.15); color: #4ade80; }
  .settings-flash.danger  { border-color: var(--crimson); background: rgba(192,57,43,.12); color: #f87171; }

  /* ── Layout ── */
  .settings-grid {
    display: grid;
    grid-template-columns: 260px 1fr;
    gap: 1.5rem;
    align-items: start;
  }

  @media (max-width: 900px) {
    .settings-grid { grid-template-columns: 1fr; }
  }

  /* ── Sidebar card ── */
  .profile-sidebar {
    background: var(--surface-1);
    border: 1px solid var(--border);
    border-radius: 16px;
    overflow: hidden;
  }

  .sidebar-top {
    background: linear-gradient(160deg, #1a0a09 0%, var(--surface-1) 100%);
    padding: 2rem 1.5rem 1.5rem;
    text-align: center;
    border-bottom: 1px solid var(--border);
    position: relative;
  }

  .sidebar-top::before {
    content: '';
    position: absolute;
    inset: 0;
    background: radial-gradient(ellipse at top center, rgba(192,57,43,.12) 0%, transparent 65%);
    pointer-events: none;
  }

  .avatar-ring {
    position: relative;
    display: inline-block;
    margin-bottom: 1rem;
  }

  .avatar-ring img {
    width: 90px;
    height: 90px;
    border-radius: 50%;
    object-fit: cover;
    display: block;
    border: 3px solid var(--crimson);
    box-shadow: 0 0 24px var(--crimson-glow), 0 0 6px rgba(0,0,0,.8);
  }

  .avatar-ring .avatar-edit-badge {
    position: absolute;
    bottom: 2px;
    right: 2px;
    width: 26px;
    height: 26px;
    background: var(--crimson);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.65rem;
    color: #fff;
    border: 2px solid var(--surface-1);
    cursor: pointer;
    transition: background .2s;
  }
  .avatar-ring .avatar-edit-badge:hover { background: var(--crimson-bright); }

   
  .sidebar-role {
    font-size: 0.75rem;
    color: var(--text-secondary);
    text-transform: uppercase;
    letter-spacing: 1px;
  }

  .sidebar-nav {
    padding: 0.75rem 0;
  }

  .sidebar-nav-item {
    display: flex;
    align-items: center;
    gap: 0.85rem;
    padding: 0.7rem 1.5rem;
    font-size: 0.875rem;
    color: var(--text-secondary);
    cursor: pointer;
    transition: all .2s;
    border-left: 3px solid transparent;
  }
  .sidebar-nav-item:hover {
    color: var(--text-primary);
    background: rgba(192,57,43,.06);
  }
  .sidebar-nav-item.active {
    color: var(--crimson-bright);
    background: rgba(192,57,43,.1);
    border-left-color: var(--crimson-bright);
  }
  .sidebar-nav-item i { width: 16px; text-align: center; }

  /* ── Main form card ── */
  .settings-card {
    background: var(--surface-1);
    border: 1px solid var(--border);
    border-radius: 16px;
    overflow: hidden;
  }

  .settings-card-header {
    padding: 1.25rem 1.75rem;
    border-bottom: 1px solid var(--border);
    display: flex;
    align-items: center;
    gap: 0.75rem;
    background: linear-gradient(90deg, rgba(192,57,43,.06) 0%, transparent 60%);
  }
   
  .settings-card-header i { color: var(--crimson-bright); }

  .settings-card-body {
    padding: 2rem 1.75rem;
  }

  /* ── Section divider ── */
  
  
  .form-section-label:first-child { margin-top: 0; }
  .form-section-label::after {
    content: '';
    flex: 1;
    height: 1px;
    background: var(--border);
  }

  /* ── Avatar upload section ── */
  .avatar-upload-row {
    display: flex;
    align-items: center;
    gap: 1.25rem;
    padding: 1.25rem;
    background: var(--surface-2);
    border: 1px solid var(--border);
    border-radius: 12px;
    margin-bottom: 1.75rem;
  }

  .avatar-upload-row img {
    width: 64px;
    height: 64px;
    border-radius: 50%;
    object-fit: cover;
    border: 2px solid var(--crimson);
    flex-shrink: 0;
  }

  .avatar-upload-info { flex: 1; }
  .avatar-upload-info label {
    display: block;
    font-size: 0.8rem;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 0.4rem;
  }

  .file-input-styled {
    position: relative;
  }
  .file-input-styled input[type="file"] {
    position: absolute;
    inset: 0;
    opacity: 0;
    cursor: pointer;
    width: 100%;
  }
  .file-input-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.45rem 1rem;
    background: var(--surface-3);
    border: 1px solid var(--border);
    border-radius: 8px;
    font-size: 0.8rem;
    color: var(--text-secondary);
    cursor: pointer;
    transition: all .2s;
    user-select: none;
  }
  .file-input-btn:hover {
    border-color: var(--border-hover);
    color: var(--text-primary);
  }
  .file-hint { font-size: 0.72rem; color: var(--text-muted); margin-top: 0.35rem; }

  /* ── Form fields ── */
  .form-group { margin-bottom: 1.25rem; }

  .form-group label {
    display: block;
    font-size: 0.8rem;
    font-weight: 500;
    color: var(--text-secondary);
    margin-bottom: 0.5rem;
    letter-spacing: 0.3px;
  }

  .form-control-dark {
    width: 100%;
    background: var(--surface-2);
    border: 1px solid var(--border);
    border-radius: 10px;
    padding: 0.7rem 1rem;
    color: var(--text-primary);
    font-family: 'DM Sans', sans-serif;
    font-size: 0.9rem;
    transition: border-color .2s, box-shadow .2s;
    outline: none;
    box-sizing: border-box;
  }
  .form-control-dark::placeholder { color: var(--text-muted); }
  .form-control-dark:focus {
    border-color: var(--crimson);
    box-shadow: 0 0 0 3px rgba(192,57,43,.15);
  }

  .input-group-dark {
    display: flex;
  }
  .input-group-dark .form-control-dark {
    border-radius: 10px 0 0 10px;
    flex: 1;
  }
  .input-group-dark .toggle-btn {
    background: var(--surface-2);
    border: 1px solid var(--border);
    border-left: none;
    border-radius: 0 10px 10px 0;
    padding: 0 1rem;
    color: var(--text-muted);
    cursor: pointer;
    transition: color .2s, background .2s;
    display: flex;
    align-items: center;
  }
  .input-group-dark .toggle-btn:hover {
    color: var(--crimson-bright);
    background: rgba(192,57,43,.08);
  }

  /* ── Two-column grid ── */
  .form-row-2 {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1.25rem;
  }
  @media (max-width: 600px) { .form-row-2 { grid-template-columns: 1fr; } }

  /* ── Divider ── */
  .section-divider {
    height: 1px;
    background: var(--border);
    margin: 1.75rem 0;
  }

  /* ── Submit button ── */
  .btn-save {
    display: inline-flex;
    align-items: center;
    gap: 0.6rem;
    padding: 0.75rem 2rem;
    background: linear-gradient(135deg, var(--crimson) 0%, #7b1a13 100%);
    border: none;
    border-radius: 10px;
    color: #fff;
    
    font-weight: 700;
    font-size: 0.875rem;
    letter-spacing: 0.5px;
    cursor: pointer;
    transition: all .2s;
    box-shadow: 0 4px 16px rgba(192,57,43,.35);
  }
  .btn-save:hover {
    transform: translateY(-1px);
    box-shadow: 0 6px 22px rgba(192,57,43,.5);
    background: linear-gradient(135deg, var(--crimson-bright) 0%, var(--crimson) 100%);
  }
  .btn-save:active { transform: translateY(0); }

  .btn-cancel {
    display: inline-flex;
    align-items: center;
    gap: 0.6rem;
    padding: 0.75rem 1.5rem;
    background: transparent;
    border: 1px solid var(--border);
    border-radius: 10px;
    color: var(--text-secondary);
    font-family: 'Syne', sans-serif;
    font-weight: 600;
    font-size: 0.875rem;
    cursor: pointer;
    transition: all .2s;
    margin-left: 0.75rem;
  }
  .btn-cancel:hover {
    border-color: var(--border-hover);
    color: var(--text-primary);
  }

  .form-actions {
    display: flex;
    align-items: center;
    padding-top: 0.5rem;
    flex-wrap: wrap;
    gap: 0.5rem;
  }

  /* ── Password strength ── */
  .pwd-strength {
    margin-top: 0.5rem;
    height: 3px;
    background: var(--surface-3);
    border-radius: 99px;
    overflow: hidden;
  }
  .pwd-strength-bar {
    height: 100%;
    width: 0%;
    border-radius: 99px;
    transition: width .3s, background .3s;
  }
  .pwd-strength-text {
    font-size: 0.72rem;
    color: var(--text-muted);
    margin-top: 0.35rem;
  }
</style>

<div class="settings-wrapper">

  <div id="top-stats-wrapper">
    <?php include __DIR__ . '/../parts/vm_top_stats.php'; ?>
  </div>

  <!-- Page Header -->
  <div class="settings-header">
    <div class="settings-header-icon">
      <i class="fa-solid fa-gear"></i>
    </div>
    <div>
      <h4>Account Settings</h4>
      <p>Manage your profile, security, and preferences</p>
    </div>
  </div>

  <?php if (!empty($_SESSION['flash'])): ?>
    <div class="settings-flash <?= $_SESSION['flash']['type'] === 'success' ? 'success' : 'danger' ?>">
      <i class="fa-solid <?= $_SESSION['flash']['type'] === 'success' ? 'fa-circle-check' : 'fa-triangle-exclamation' ?>"></i>
      <?= htmlspecialchars($_SESSION['flash']['msg']) ?>
    </div>
    <?php unset($_SESSION['flash']); ?>
  <?php endif; ?>

  <div class="settings-grid">

    <!-- Sidebar -->
    <aside class="profile-sidebar">
      <div class="sidebar-top">
        <div class="avatar-ring">
          <img src="<?= htmlspecialchars($avatarUrl) ?>" alt="Avatar" id="sidebar-avatar-preview">
          <div class="avatar-edit-badge" onclick="document.getElementById('avatar-file-input').click()">
            <i class="fa-solid fa-pen"></i>
          </div>
        </div>
        <div class="sidebar-username"><?= htmlspecialchars($currentUsername) ?></div>
        <div class="sidebar-role">Member</div>
      </div>
      <nav class="sidebar-nav">
        <div class="sidebar-nav-item active">
          <i class="fa-solid fa-user-pen"></i> Profile 
        </div>
  
      </nav>
    </aside>

    <!-- Main Card -->
    <div class="settings-card">

      <div class="settings-card-header">
        <i class="fa-solid fa-user-pen"></i>
        <span>Profile & Password</span>
      </div>

      <div class="settings-card-body">
        <form method="post" action="/earnapp/change_password.php" enctype="multipart/form-data" id="settings-form">

          <!-- Avatar Upload -->
          <div class="form-section-label"><i class="fa-solid fa-image"></i> Profile Photo</div>

          <div class="avatar-upload-row">
            <img src="<?= htmlspecialchars($avatarUrl) ?>" alt="Avatar" id="main-avatar-preview">
            <div class="avatar-upload-info">
              <label>Upload a new photo</label>
              <div class="file-input-styled">
                <div class="file-input-btn" id="file-btn-label">
                  <i class="fa-solid fa-arrow-up-from-bracket"></i>
                  <span id="file-btn-text">Choose file…</span>
                </div>
                <input type="file" name="avatar" id="avatar-file-input"
                       accept="image/png,image/jpeg,image/webp"
                       onchange="handleFileChange(this)">
              </div>
              <div class="file-hint">PNG, JPG or WEBP &mdash; max 2 MB</div>
            </div>
          </div>

          <!-- Profile Info -->
          <div class="form-section-label"><i class="fa-solid fa-id-badge"></i> Profile Info</div>

          <div class="form-group">
            <label for="username">Username</label>
            <input type="text"
                   id="username"
                   name="username"
                   value="<?= htmlspecialchars($currentUsername) ?>"
                   class="form-control-dark"
                   required>
          </div>

          <div class="section-divider"></div>

          <!-- Security -->
          <div class="form-section-label"><i class="fa-solid fa-lock"></i> Change Password</div>

          <div class="form-row-2">
            <div class="form-group">
              <label for="new_password">New Password</label>
              <div class="input-group-dark">
                <input type="password"
                       id="new_password"
                       name="new_password"
                       class="form-control-dark"
                       placeholder="Leave blank to keep current"
                       oninput="checkStrength(this.value)">
                <button type="button" class="toggle-btn" onclick="togglePwd('new_password', this)">
                  <i class="fa-solid fa-eye"></i>
                </button>
              </div>
              <div class="pwd-strength">
                <div class="pwd-strength-bar" id="pwd-bar"></div>
              </div>
              <div class="pwd-strength-text" id="pwd-text"></div>
            </div>

            <div class="form-group">
              <label for="confirm_password">Confirm New Password</label>
              <div class="input-group-dark">
                <input type="password"
                       id="confirm_password"
                       name="confirm_password"
                       class="form-control-dark"
                       placeholder="Re-enter new password">
                <button type="button" class="toggle-btn" onclick="togglePwd('confirm_password', this)">
                  <i class="fa-solid fa-eye"></i>
                </button>
              </div>
            </div>
          </div>

          <!-- Actions -->
          <div class="section-divider"></div>
          <div class="form-actions">
            <button type="submit" class="btn-save">
              <i class="fa-solid fa-floppy-disk"></i> Save Changes
            </button>
            <button type="reset" class="btn-cancel">
              Discard
            </button>
          </div>

        </form>
      </div>
    </div>

  </div>
</div>

<script>
function togglePwd(id, btn) {
  const input = document.getElementById(id);
  const icon = btn.querySelector('i');
  if (input.type === 'password') {
    input.type = 'text';
    icon.classList.replace('fa-eye', 'fa-eye-slash');
  } else {
    input.type = 'password';
    icon.classList.replace('fa-eye-slash', 'fa-eye');
  }
}

function handleFileChange(input) {
  if (!input.files.length) return;
  const file = input.files[0];
  const url = URL.createObjectURL(file);

  document.getElementById('main-avatar-preview').src = url;
  document.getElementById('sidebar-avatar-preview').src = url;
  document.getElementById('file-btn-text').textContent = file.name.length > 22
    ? file.name.slice(0, 20) + '…'
    : file.name;
}

function checkStrength(val) {
  const bar = document.getElementById('pwd-bar');
  const txt = document.getElementById('pwd-text');
  if (!val) { bar.style.width = '0%'; txt.textContent = ''; return; }

  let score = 0;
  if (val.length >= 8) score++;
  if (/[A-Z]/.test(val)) score++;
  if (/[0-9]/.test(val)) score++;
  if (/[^A-Za-z0-9]/.test(val)) score++;

  const levels = [
    { w: '25%', bg: '#e74c3c', label: 'Weak' },
    { w: '50%', bg: '#e67e22', label: 'Fair' },
    { w: '75%', bg: '#f1c40f', label: 'Good' },
    { w: '100%', bg: '#2ecc71', label: 'Strong' },
  ];
  const l = levels[score - 1] || levels[0];
  bar.style.width = l.w;
  bar.style.background = l.bg;
  txt.textContent = l.label;
  txt.style.color = l.bg;
}
</script>