<?php
include '../db.php'; // Make sure this path is correct
$conn->set_charset("utf8mb4");
session_start();

if(!isset($_SESSION['user_id'])){
    header('Location: login.php');
    exit;
}

// Dummy user data for display, replace with your actual user fetching logic
$user_id = $_SESSION['user_id'];
// Example: $result = $conn->query("SELECT full_name FROM users WHERE id = $user_id");
// $user = $result->fetch_assoc();
$user = ['full_name' => 'Admin']; // Placeholder

// --- Countries Management Logic --- //

$message = ''; // To store success/error messages
$country_to_edit = null; // To store data for the country being edited

// Handle POST Request (When the edit form is submitted)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action']) && $_POST['action'] == 'update_country') {
    $country_code = $_POST['country_code'];
    $country_name = $_POST['country_name'];
    $idp_convention = $_POST['idp_convention'];

    if (!empty($country_code) && !empty($country_name)) {
        // Use prepared statement to prevent SQL injection
        $sql = "UPDATE countries SET country_name = ?, idp_convention = ? WHERE country_code = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("sss", $country_name, $idp_convention, $country_code);

        if ($stmt->execute()) {
            $message = "Country updated successfully!";
        } else {
            $message = "Error updating country: " . $stmt->error;
        }
        $stmt->close();
    } else {
        $message = "Country name cannot be empty.";
    }
}

// --- NEW: Handle POST Request (When the remove form is submitted) --- //
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action']) && $_POST['action'] == 'delete_country') {
    $country_code_to_delete = $_POST['country_code'];

    if (!empty($country_code_to_delete)) {
        // Use prepared statement to prevent SQL injection
        $sql = "DELETE FROM countries WHERE country_code = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("s", $country_code_to_delete);

        if ($stmt->execute()) {
            $message = "Country removed successfully!";
        } else {
            $message = "Error removing country: " . $stmt->error;
        }
        $stmt->close();
    } else {
        $message = "Invalid country code for deletion.";
    }
}

// Handle GET Request (When "Edit" link is clicked)
if (isset($_GET['edit'])) {
    $country_code_to_edit = $_GET['edit'];
    
    $sql = "SELECT country_code, country_name, idp_convention FROM countries WHERE country_code = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $country_code_to_edit);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows == 1) {
        $country_to_edit = $result->fetch_assoc();
    }
    $stmt->close();
}

// Fetch all countries to display in the table
$countries = [];
$sql = "SELECT country_code, country_name, idp_convention FROM countries ORDER BY country_name ASC";
$result = $conn->query($sql);
if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $countries[] = $row;
    }
}

// --- NEW: Get the count of countries --- //
$country_count = count($countries);
// --- End of Countries Logic --- //

?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Manage Countries Dashboard</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<link rel="stylesheet" href="assets/css/style.css"> <!-- Make sure this path is correct -->
<style>
    .circle-name { 
        width: 40px; height: 40px; 
        border-radius: 50%; background: #007bff; 
        color: #fff; text-align: center; line-height: 40px; 
        font-weight: bold; margin-right: 10px;
    }
    .edit-form-container {
        background-color: #f8f9fa;
        padding: 2rem;
        border-radius: 0.5rem;
        margin-bottom: 2rem;
        border: 1px solid #dee2e6;
    }
    .action-buttons form {
        display: inline-block;
        margin-left: 5px;
    }
</style>
</head>
<body>
<div class="dashboard_wrapper">
    <div class="left_panel">
        <div class="logo_box">
            <a href="/"><img src="../assets/images/logo.png" alt="logo" class="img-fluid full_logo"></a>
            <a href="/"><img src="../assets/images/logo-icon.png" alt="logo" class="img-fluid logo_icon"></a>
        </div>
        <?php include 'menu.php'; ?>
    </div>

    <div class="right_body_panel">
        <div class="right_header_panel">
            <div class="left_toggle_button">
                <button class="btn toggle_btn" type="button"></button>
            </div>
            <div class="right_user_box">
                <div class="user_box_warp">
                    <span class="circle-name"><?php echo strtoupper(substr($user['full_name'], 0, 2)); ?></span>
                    <div class="user_name"><h4><?php echo htmlspecialchars($user['full_name']); ?></h4></div>
                </div>
            </div>
        </div>

        <div class="body_wrapper_section p-3">
            <div class="body_wrap">
                <!-- MODIFIED: Added country count badge -->
                <h3>Manage Countries <span class="badge bg-secondary"><?php echo $country_count; ?></span></h3>
                <div class="inner-wrapper">
                    
                    <?php if (!empty($message)): ?>
                        <div class="alert alert-info alert-dismissible fade show" role="alert">
                            <?php echo htmlspecialchars($message); ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                        </div>
                    <?php endif; ?>

                    <!-- Edit Form (shows only when editing) -->
                    <?php if ($country_to_edit): ?>
                    <div class="edit-form-container">
                        <h4>Editing: <?php echo htmlspecialchars($country_to_edit['country_name']); ?></h4>
                        <form action="countries.php" method="post" class="mt-3">
                            <input type="hidden" name="action" value="update_country">
                            <input type="hidden" name="country_code" value="<?php echo htmlspecialchars($country_to_edit['country_code']); ?>">
                            
                            <div class="mb-3">
                                <label for="country_name" class="form-label">Country Name</label>
                                <input type="text" id="country_name" name="country_name" class="form-control" value="<?php echo htmlspecialchars($country_to_edit['country_name']); ?>" required>
                            </div>
                            
                            <div class="mb-3">
                                <label for="idp_convention" class="form-label">Accepted IDP Format/Convention</label>
                                <textarea id="idp_convention" name="idp_convention" class="form-control" rows="3"><?php echo htmlspecialchars($country_to_edit['idp_convention']); ?></textarea>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">Save Changes</button>
                            <a href="countries.php" class="btn btn-secondary">Cancel</a>
                        </form>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Countries List Table -->
                    <div class="table-responsive">
                        <table class="table table-striped table-bordered">
                            <thead class="table-dark">
                                <tr>
                                    <th>Country Name</th>
                                    <th>IDP Convention</th>
                                    <!-- MODIFIED: Increased width for more buttons -->
                                    <th style="width: 180px;">Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($countries as $country): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($country['country_name']); ?></td>
                                    <td><?php echo htmlspecialchars($country['idp_convention']); ?></td>
                                    <!-- MODIFIED: Added remove button inside a form -->
                                    <td class="action-buttons">
                                        <a href="countries.php?edit=<?php echo urlencode($country['country_code']); ?>" class="btn btn-sm btn-primary">Edit</a>
                                        <form action="countries.php" method="post">
                                            <input type="hidden" name="action" value="delete_country">
                                            <input type="hidden" name="country_code" value="<?php echo htmlspecialchars($country['country_code']); ?>">
                                            <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this country?');">Remove</button>
                                        </form>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Optional: Add script for your toggle button if needed
document.querySelector('.toggle_btn').addEventListener('click', function() {
    document.querySelector('.dashboard_wrapper').classList.toggle('is-active');
});
</script>

</body>
</html>
<?php
$conn->close();
?>