<?php
include 'session.php';
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(["status" => "error", "message" => "Invalid request method"]);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents("php://input"), true);

// Validate receiver data
$requiredFields = ['recFirstName', 'recLastName', 'postalCode', 'cityName', 'countryCode', 'recState', 'addressLine1'];
foreach ($requiredFields as $field) {
    if (empty($input['receiverDetails'][$field])) {
        echo json_encode(["status" => "error", "errors" => [$field => "This field is required."]]);
        exit;
    }
}

$receiver = $input['receiverDetails'];

// Check if session has origin address
if (!isset($_SESSION['step1'])) {
    echo json_encode(["status" => "error", "message" => "Missing origin address from session"]);
    exit;
}

$origin = $_SESSION['step1'];

// --- Fetch city/state using DHL Location Finder API ---
$city = $receiver['cityName'];
$state = $receiver['recState'];

$dhlKey = 'pAhLmsinCbAg43j0Cn7PKVkZZy4BO41l';
$dhlEndpoint = "https://api.dhl.com/location-finder/v1/find-by-address?countryCode={$receiver['countryCode']}&postalCode={$receiver['postalCode']}";

$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => $dhlEndpoint,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER => [
        "DHL-API-Key: $dhlKey",
        "Accept: application/json"
    ],
    CURLOPT_TIMEOUT => 5,
]);
$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($httpCode === 200) {
    $data = json_decode($response, true);
    if (!empty($data['locations'])) {
        $location = $data['locations'][0]['place']['address'] ?? [];
        if (!$city) $city = $location['addressLocality'] ?? '';
        if (!$state) $state = $location['addressAdministrativeArea'] ?? '';
    }
}

// --- Fallback for India only ---
if (strtoupper($receiver['countryCode']) === 'IN') {
    if (!$city || !$state) {
        $fallbackUrl = "http://www.postalpincode.in/api/pincode/{$receiver['postalCode']}";
        $ch2 = curl_init();
        curl_setopt_array($ch2, [
            CURLOPT_URL => $fallbackUrl,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 5,
        ]);
        $fallbackResp = curl_exec($ch2);
        curl_close($ch2);

        $fData = json_decode($fallbackResp, true);
        if (!empty($fData['PostOffice'][0])) {
            if (!$city) $city = $fData['PostOffice'][0]['District'] ?? '';
            if (!$state) $state = $fData['PostOffice'][0]['State'] ?? '';
        }
    }
}

// --- Prepare DHL API request ---
$requestData = [
    "customerDetails" => [
        "shipperDetails" => [
            "postalCode"   => $origin['zip'],
            "cityName"     => $origin['city'],
            "countryCode"  => substr($origin['country'], 0, 2),
            "addressLine1" => $origin['street']
        ],
        "receiverDetails" => [
            "postalCode"   => $receiver['postalCode'],
            "cityName"     => $city ?? "",   // send empty string if city not found
            "countryCode"  => substr($receiver['countryCode'], 0, 2),
            "addressLine1" => $receiver['addressLine1']
        ]
    ],
    "accounts" => [
        [
            "typeCode" => "shipper",
            "number"   => "425959129"
        ]
    ],
    "productCode" => "P", // optionally, validate your product codes
    // Dynamically adjust pickup date to reduce 404 errors
    "plannedShippingDateAndTime" => date('c', strtotime('+2 days 09:00:00')),
    "unitOfMeasurement" => "metric",
    "isCustomsDeclarable" => true,
    "monetaryAmount" => [
        [
            "typeCode" => "declaredValue",
            "value"    => 100,
            "currency" => "USD"
        ]
    ],
    "packages" => [
        [
            "weight" => 0.8,
            "dimensions" => [
                "length" => 30,
                "width"  => 22,
                "height" => 2
            ]
        ]
    ]
];

// --- Send request to DHL ---
try {
    $rates = getDhlRates($requestData);

    if (!isset($rates["products"]) || empty($rates["products"])) {
        echo json_encode([
            "status" => "error",
            "errors" => ["dhl" => "No products returned. Check your request or product code."]
        ]);
        exit;
    }

    echo json_encode([
        "status" => "success",
        "data" => $rates
    ]);
} catch (Exception $e) {
    // Catch 404 or 422 errors gracefully
    echo json_encode([
        "status" => "error",
        "message" => $e->getMessage()
    ]);
}

// --- DHL API call function ---
function getDhlRates(array $data): array
{
    $endpoint = "https://express.api.dhl.com/mydhlapi/test/rates";
    $authToken = "Basic YXBKOG1ONHhONXlZNXI6QSQwc1UhMWdTJDd3TiE2bg==";

    $payload = json_encode($data);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $endpoint,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_CUSTOMREQUEST => "POST",
        CURLOPT_POSTFIELDS => $payload,
        CURLOPT_HTTPHEADER => [
            "Authorization: $authToken",
            "Content-Type: application/json"
        ]
    ]);

    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        throw new Exception("cURL Error: " . curl_error($ch));
    }

    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Return response even for 404/422 to handle gracefully
    if ($httpCode !== 200) {
        throw new Exception("DHL API returned HTTP code $httpCode. Response: $response");
    }

    return json_decode($response, true);
}
?>
